<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');

require_once '../../config/database.php';

try {
    $database = new Database();
    $db = $database->getConnection();

    // Get user activity for the last 7 days
    $userActivityQuery = "
        SELECT 
            DATE(us.session_start) as date,
            DAYNAME(us.session_start) as day_name,
            COUNT(DISTINCT us.user_id) as active_users
        FROM user_sessions us 
        WHERE us.session_start >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY DATE(us.session_start), DAYNAME(us.session_start)
        ORDER BY DATE(us.session_start)
    ";

    $stmt = $db->prepare($userActivityQuery);
    $stmt->execute();
    $userActivity = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Fill missing days with 0 values
    $days = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
    $activityData = [];
    $activityLabels = [];

    for ($i = 6; $i >= 0; $i--) {
        $date = date('Y-m-d', strtotime("-$i days"));
        $dayName = date('l', strtotime("-$i days"));
        $shortDay = date('D', strtotime("-$i days"));

        $found = false;
        foreach ($userActivity as $activity) {
            if ($activity['date'] === $date) {
                $activityData[] = (int)$activity['active_users'];
                $found = true;
                break;
            }
        }

        if (!$found) {
            $activityData[] = 0;
        }

        $activityLabels[] = $shortDay;
    }

    // Get revenue distribution
    $revenueQuery = "
        SELECT 
            'Game Earnings' as type,
            COALESCE(SUM(us.earnings_generated), 0) as amount
        FROM user_sessions us
        WHERE us.session_start >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        
        UNION ALL
        
        SELECT 
            'AdMob Revenue' as type,
            COALESCE(SUM(us.admob_revenue), 0) as amount
        FROM user_sessions us
        WHERE us.session_start >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        
        UNION ALL
        
        SELECT 
            'Withdrawals' as type,
            COALESCE(SUM(wr.amount), 0) as amount
        FROM withdrawal_requests wr
        WHERE wr.request_date >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        AND wr.status = 'completed'
    ";

    $stmt = $db->prepare($revenueQuery);
    $stmt->execute();
    $revenueData = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Prepare revenue chart data
    $revenueLabels = [];
    $revenueAmounts = [];
    $revenueColors = ['#10B981', '#F59E0B', '#EF4444'];

    foreach ($revenueData as $index => $revenue) {
        $revenueLabels[] = $revenue['type'];
        $revenueAmounts[] = (float)$revenue['amount'];
    }

    // Get additional metrics
    $metricsQuery = "
        SELECT 
            COUNT(DISTINCT u.id) as total_users,
            COUNT(DISTINCT CASE WHEN u.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN u.id END) as new_users_week,
            COALESCE(SUM(u.current_balance), 0) as total_user_balance,
            COUNT(DISTINCT CASE WHEN us.session_start >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN us.user_id END) as daily_active_users,
            COALESCE(AVG(TIMESTAMPDIFF(MINUTE, us.session_start, us.session_end)), 0) as avg_session_duration,
            COUNT(us.id) as total_sessions,
            COUNT(CASE WHEN us.race_completed = 1 THEN 1 END) as completed_sessions
        FROM users u
        LEFT JOIN user_sessions us ON u.id = us.user_id
        WHERE u.status = 'active'
    ";

    $stmt = $db->prepare($metricsQuery);
    $stmt->execute();
    $metrics = $stmt->fetch(PDO::FETCH_ASSOC);

    // Calculate completion rate
    $completionRate = $metrics['total_sessions'] > 0 ?
        round(($metrics['completed_sessions'] / $metrics['total_sessions']) * 100, 1) : 0;

    // Get playtime distribution
    $playtimeQuery = "
        SELECT 
            CASE 
                WHEN total_playtime_minutes < 60 THEN '< 1 hour'
                WHEN total_playtime_minutes < 300 THEN '1-5 hours'
                WHEN total_playtime_minutes < 600 THEN '5-10 hours'
                WHEN total_playtime_minutes < 1200 THEN '10-20 hours'
                ELSE '20+ hours'
            END as playtime_range,
            COUNT(*) as user_count
        FROM users 
        WHERE status = 'active'
        GROUP BY 
            CASE 
                WHEN total_playtime_minutes < 60 THEN '< 1 hour'
                WHEN total_playtime_minutes < 300 THEN '1-5 hours'
                WHEN total_playtime_minutes < 600 THEN '5-10 hours'
                WHEN total_playtime_minutes < 1200 THEN '10-20 hours'
                ELSE '20+ hours'
            END
        ORDER BY MIN(total_playtime_minutes)
    ";

    $stmt = $db->prepare($playtimeQuery);
    $stmt->execute();
    $playtimeData = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $playtimeLabels = [];
    $playtimeCounts = [];

    foreach ($playtimeData as $data) {
        $playtimeLabels[] = $data['playtime_range'];
        $playtimeCounts[] = (int)$data['user_count'];
    }

    echo json_encode([
        'success' => true,
        'data' => [
            'user_activity' => [
                'labels' => $activityLabels,
                'data' => $activityData
            ],
            'revenue_distribution' => [
                'labels' => $revenueLabels,
                'data' => $revenueAmounts,
                'colors' => $revenueColors
            ],
            'playtime_distribution' => [
                'labels' => $playtimeLabels,
                'data' => $playtimeCounts
            ],
            'metrics' => [
                'total_users' => (int)$metrics['total_users'],
                'new_users_week' => (int)$metrics['new_users_week'],
                'total_user_balance' => (float)$metrics['total_user_balance'],
                'daily_active_users' => (int)$metrics['daily_active_users'],
                'avg_session_duration' => round((float)$metrics['avg_session_duration'], 1),
                'total_sessions' => (int)$metrics['total_sessions'],
                'completion_rate' => $completionRate
            ]
        ]
    ]);
} catch (Exception $e) {
    error_log("Analytics Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to load analytics data: ' . $e->getMessage()
    ]);
}
