// Admin Dashboard JavaScript

// Global variables
let currentSection = 'dashboard';
let charts = {};

// Utility function to destroy all existing charts
function destroyAllCharts() {
    for (let chartKey in charts) {
        if (charts[chartKey] && typeof charts[chartKey].destroy === 'function') {
            charts[chartKey].destroy();
            charts[chartKey] = null;
        }
    }

    // Also clear any Chart.js registry issues
    const canvasIds = ['earningsChart', 'withdrawalChart', 'activityChart', 'revenueChart', 'playtimeChart'];
    canvasIds.forEach(id => {
        const canvas = document.getElementById(id);
        if (canvas) {
            const ctx = canvas.getContext('2d');
            ctx.clearRect(0, 0, canvas.width, canvas.height);
        }
    });
}

// Initialize dashboard
document.addEventListener('DOMContentLoaded', function () {
    initializeEventListeners();
    initializeSidebar();
    loadDashboardData();
    showSection('dashboard', null);
});

function initializeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');

    // Initialize sidebar state based on screen size
    if (window.innerWidth >= 1024) {
        // Desktop: show sidebar
        sidebar.classList.remove('-translate-x-full');
        sidebar.classList.add('translate-x-0');
        overlay.classList.remove('active');
    } else {
        // Mobile: hide sidebar
        sidebar.classList.add('-translate-x-full');
        sidebar.classList.remove('translate-x-0');
        overlay.classList.remove('active');
    }

    // Handle window resize
    window.addEventListener('resize', function () {
        if (window.innerWidth >= 1024) {
            // Desktop: show sidebar
            sidebar.classList.remove('-translate-x-full');
            sidebar.classList.add('translate-x-0');
            overlay.classList.remove('active');
        } else {
            // Mobile: hide sidebar
            sidebar.classList.add('-translate-x-full');
            sidebar.classList.remove('translate-x-0');
            overlay.classList.remove('active');
        }
    });
}

function toggleSidebar() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');

    if (window.innerWidth < 1024) {
        // Mobile behavior only
        const isVisible = sidebar.classList.contains('translate-x-0');

        if (isVisible) {
            // Hide sidebar
            sidebar.classList.add('-translate-x-full');
            sidebar.classList.remove('translate-x-0');
            overlay.classList.remove('active');
        } else {
            // Show sidebar
            sidebar.classList.remove('-translate-x-full');
            sidebar.classList.add('translate-x-0');
            overlay.classList.add('active');
        }
    }
}

function closeSidebar() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('sidebar-overlay');

    if (window.innerWidth < 1024) {
        // Hide sidebar on mobile
        sidebar.classList.add('-translate-x-full');
        sidebar.classList.remove('translate-x-0');
        overlay.classList.remove('active');
    }
}

function initializeEventListeners() {
    // Sidebar toggle
    document.getElementById('sidebar-toggle').addEventListener('click', function () {
        toggleSidebar();
    });

    // Status filter change
    document.getElementById('status-filter').addEventListener('change', function () {
        loadWithdrawals();
    });

    // Settings form submit
    document.getElementById('settings-form').addEventListener('submit', function (e) {
        e.preventDefault();
        saveSettings();
    });

    // Withdrawal form submit
    document.getElementById('withdrawal-form').addEventListener('submit', function (e) {
        e.preventDefault();
        processWithdrawal();
    });
}

function showSection(section, targetElement = null) {
    // Destroy all existing charts to prevent canvas reuse errors
    destroyAllCharts();

    // Hide all sections
    document.querySelectorAll('.section-content').forEach(el => {
        el.classList.add('hidden');
    });

    // Remove active class from all nav links
    document.querySelectorAll('.nav-link').forEach(el => {
        el.classList.remove('bg-gray-700', 'text-white');
        el.classList.add('text-gray-300');
    });

    // Show selected section
    document.getElementById(section + '-section').classList.remove('hidden');

    // Add active class to selected nav link
    if (targetElement) {
        targetElement.classList.add('bg-gray-700', 'text-white');
        targetElement.classList.remove('text-gray-300');
    } else {
        // If no target element, find and activate the corresponding nav link
        const navLinks = document.querySelectorAll('.nav-link');
        navLinks.forEach(link => {
            if (link.onclick && link.onclick.toString().includes(`'${section}'`)) {
                link.classList.add('bg-gray-700', 'text-white');
                link.classList.remove('text-gray-300');
            }
        });
    }

    // Close sidebar on mobile when menu item is selected
    if (window.innerWidth < 1024) {
        closeSidebar();
    }

    // Update page title
    const titles = {
        'dashboard': 'Dashboard',
        'withdrawals': 'Withdrawal Requests',
        'users': 'User Management',
        'analytics': 'Analytics',
        'settings': 'Settings'
    };
    document.getElementById('page-title').textContent = titles[section];

    currentSection = section;

    // Load section-specific data
    switch (section) {
        case 'dashboard':
            loadDashboardData();
            break;
        case 'withdrawals':
            loadWithdrawals();
            break;
        case 'users':
            loadUsers();
            break;
        case 'analytics':
            loadAnalytics();
            break;
        case 'settings':
            loadSettings();
            break;
    }
}

async function loadDashboardData() {
    try {
        // Load dashboard statistics
        const response = await fetch('api/admin_stats.php');

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();

        if (data.success) {
            const totalUsers = document.getElementById('total-users');
            const pendingWithdrawals = document.getElementById('pending-withdrawals');
            const totalPayouts = document.getElementById('total-payouts');
            const totalBalance = document.getElementById('total-balance');

            if (totalUsers) totalUsers.textContent = data.stats.total_users || '0';
            if (pendingWithdrawals) pendingWithdrawals.textContent = data.stats.pending_withdrawals || '0';
            if (totalPayouts) totalPayouts.textContent = '$' + (data.stats.total_payouts || '0');
            if (totalBalance) totalBalance.textContent = '$' + (data.stats.total_balance || '0');

            // Initialize charts
            initializeCharts(data.chart_data);
        } else {
            throw new Error(data.message || 'Failed to load dashboard data');
        }
    } catch (error) {
        console.error('Error loading dashboard data:', error);
        showNotification('Error loading dashboard data: ' + error.message, 'error');

        // Set default values if API fails
        const totalUsers = document.getElementById('total-users');
        const pendingWithdrawals = document.getElementById('pending-withdrawals');
        const totalPayouts = document.getElementById('total-payouts');
        const totalBalance = document.getElementById('total-balance');

        if (totalUsers) totalUsers.textContent = '0';
        if (pendingWithdrawals) pendingWithdrawals.textContent = '0';
        if (totalPayouts) totalPayouts.textContent = '$0';
        if (totalBalance) totalBalance.textContent = '$0';

        // Initialize empty charts
        initializeCharts({
            earnings_labels: [],
            earnings_data: [],
            withdrawal_status: [0, 0, 0, 0]
        });
    }
}

async function loadWithdrawals() {
    try {
        const statusFilter = document.getElementById('status-filter').value;
        const url = `api/admin_withdrawals.php${statusFilter ? '?status=' + statusFilter : ''}`;

        const response = await fetch(url);

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const data = await response.json();

        if (data.success) {
            const tbody = document.getElementById('withdrawals-table');
            tbody.innerHTML = '';

            data.withdrawals.forEach(withdrawal => {
                const row = createWithdrawalRow(withdrawal);
                tbody.appendChild(row);
            });
        } else {
            throw new Error(data.message || 'Failed to load withdrawals');
        }
    } catch (error) {
        console.error('Error loading withdrawals:', error);
        showNotification('Error loading withdrawal requests: ' + error.message, 'error');

        // Clear table on error
        const tbody = document.getElementById('withdrawals-table');
        if (tbody) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center py-4 text-gray-500">Error loading data</td></tr>';
        }
    }
}

function createWithdrawalRow(withdrawal) {
    const row = document.createElement('tr');

    const statusClass = {
        'pending': 'bg-yellow-100 text-yellow-800',
        'approved': 'bg-blue-100 text-blue-800',
        'completed': 'bg-green-100 text-green-800',
        'rejected': 'bg-red-100 text-red-800'
    };

    row.innerHTML = `
        <td class="px-6 py-4 whitespace-nowrap">
            <div class="text-sm font-medium text-gray-900">${withdrawal.username}</div>
            <div class="text-sm text-gray-500">${withdrawal.email}</div>
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
            $${parseFloat(withdrawal.amount).toFixed(2)}
        </td>
        <td class="px-6 py-4 whitespace-nowrap">
            <div class="text-sm text-gray-900">${withdrawal.payment_type}</div>
            <div class="text-sm text-gray-500">${withdrawal.payment_address}</div>
        </td>
        <td class="px-6 py-4 whitespace-nowrap">
            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full ${statusClass[withdrawal.status]}">
                ${withdrawal.status}
            </span>
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
            ${new Date(withdrawal.request_date).toLocaleDateString()}
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
            ${withdrawal.status === 'pending' ?
            `<button onclick="openWithdrawalModal(${withdrawal.id})" class="text-indigo-600 hover:text-indigo-900">Process</button>` :
            `<span class="text-gray-400">Processed</span>`
        }
        </td>
    `;

    return row;
}

async function loadUsers() {
    try {
        const response = await fetch('api/admin_users.php');
        const data = await response.json();

        if (data.success) {
            const tbody = document.getElementById('users-table');
            tbody.innerHTML = '';

            data.users.forEach(user => {
                const row = createUserRow(user);
                tbody.appendChild(row);
            });
        }
    } catch (error) {
        console.error('Error loading users:', error);
        showNotification('Error loading users', 'error');
    }
}

function createUserRow(user) {
    const row = document.createElement('tr');

    const statusClass = {
        'active': 'bg-green-100 text-green-800',
        'suspended': 'bg-yellow-100 text-yellow-800',
        'banned': 'bg-red-100 text-red-800'
    };

    row.innerHTML = `
        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
            ${user.username}
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
            ${user.email}
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
            $${parseFloat(user.current_balance).toFixed(2)}
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
            $${parseFloat(user.total_earned).toFixed(2)}
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
            ${Math.floor(user.total_playtime_minutes / 60)}h ${user.total_playtime_minutes % 60}m
        </td>
        <td class="px-6 py-4 whitespace-nowrap">
            <span class="inline-flex px-2 py-1 text-xs font-semibold rounded-full ${statusClass[user.status]}">
                ${user.status}
            </span>
        </td>
        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
            <button class="text-indigo-600 hover:text-indigo-900 mr-2">Edit</button>
            <button class="text-red-600 hover:text-red-900">Suspend</button>
        </td>
    `;

    return row;
}

function initializeCharts(chartData) {
    // Ensure dashboard section is visible before creating charts
    const dashboardSection = document.getElementById('dashboard-section');
    if (!dashboardSection || dashboardSection.classList.contains('hidden')) {
        return;
    }

    // Earnings Chart
    const earningsCtx = document.getElementById('earningsChart');
    if (!earningsCtx) return;

    if (charts.earnings) charts.earnings.destroy();

    charts.earnings = new Chart(earningsCtx.getContext('2d'), {
        type: 'line',
        data: {
            labels: chartData.earnings_labels || [],
            datasets: [{
                label: 'Daily Earnings',
                data: chartData.earnings_data || [],
                borderColor: 'rgb(59, 130, 246)',
                backgroundColor: 'rgba(59, 130, 246, 0.1)',
                tension: 0.1
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        callback: function (value) {
                            return '$' + value;
                        }
                    }
                }
            },
            plugins: {
                legend: {
                    display: false
                }
            }
        }
    });

    // Withdrawal Status Chart
    const withdrawalCtx = document.getElementById('withdrawalChart');
    if (!withdrawalCtx) return;

    if (charts.withdrawal) charts.withdrawal.destroy();

    charts.withdrawal = new Chart(withdrawalCtx.getContext('2d'), {
        type: 'doughnut',
        data: {
            labels: ['Pending', 'Approved', 'Completed', 'Rejected'],
            datasets: [{
                data: chartData.withdrawal_status || [0, 0, 0, 0],
                backgroundColor: [
                    '#FEF3C7',
                    '#DBEAFE',
                    '#D1FAE5',
                    '#FEE2E2'
                ],
                borderColor: [
                    '#F59E0B',
                    '#3B82F6',
                    '#10B981',
                    '#EF4444'
                ]
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: true,
            aspectRatio: 1,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 15,
                        usePointStyle: true,
                        font: {
                            size: 11
                        }
                    }
                }
            },
            layout: {
                padding: {
                    top: 10,
                    bottom: 10,
                    left: 10,
                    right: 10
                }
            }
        }
    });
}

function openWithdrawalModal(withdrawalId) {
    document.getElementById('withdrawal-id').value = withdrawalId;
    document.getElementById('withdrawal-modal').classList.remove('hidden');
}

function closeModal() {
    document.getElementById('withdrawal-modal').classList.add('hidden');
}

async function processWithdrawal() {
    try {
        const formData = new FormData(document.getElementById('withdrawal-form'));

        const response = await fetch('api/process_withdrawal.php', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            showNotification('Withdrawal processed successfully', 'success');
            closeModal();
            loadWithdrawals();
            loadDashboardData(); // Refresh stats
        } else {
            showNotification(data.message || 'Error processing withdrawal', 'error');
        }
    } catch (error) {
        console.error('Error processing withdrawal:', error);
        showNotification('Error processing withdrawal', 'error');
    }
}

async function loadSettings() {
    try {
        const response = await fetch('api/admin_settings.php');
        const data = await response.json();

        if (data.success) {
            const form = document.getElementById('settings-form');
            data.settings.forEach(setting => {
                const input = form.querySelector(`[name="${setting.setting_key}"]`);
                if (input) {
                    if (input.type === 'checkbox') {
                        input.checked = setting.setting_value === '1' || setting.setting_value === 'true';
                    } else {
                        input.value = setting.setting_value;
                    }
                }
            });
        }
    } catch (error) {
        console.error('Error loading settings:', error);
        showNotification('Error loading settings', 'error');
    }
}

async function saveSettings() {
    try {
        const form = document.getElementById('settings-form');
        const formData = new FormData(form);

        // Handle checkbox for withdrawals_enabled
        const withdrawalsEnabled = form.querySelector('[name="withdrawals_enabled"]');
        if (withdrawalsEnabled) {
            formData.set('withdrawals_enabled', withdrawalsEnabled.checked ? '1' : '0');
        }

        const response = await fetch('api/admin_settings.php', {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success) {
            showNotification('Settings saved successfully', 'success');
        } else {
            showNotification(data.message || 'Error saving settings', 'error');
        }
    } catch (error) {
        console.error('Error saving settings:', error);
        showNotification('Error saving settings', 'error');
    }
}

function loadAnalytics() {
    console.log('loadAnalytics() called'); // Debug log

    // Show loading state
    showLoadingState('analytics');

    // Fetch analytics data from API
    console.log('Fetching analytics data...'); // Debug log
    fetch('./api/admin_analytics.php')
        .then(response => {
            console.log('Analytics response received:', response); // Debug log
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Analytics data:', data); // Debug log
            if (data.success) {
                updateAnalyticsMetrics(data.data.metrics);
                initializeAnalyticsCharts(data.data);
            } else {
                console.error('Failed to load analytics:', data.message);
                showNotification('Failed to load analytics data', 'error');
            }
        })
        .catch(error => {
            console.error('Error loading analytics:', error);
            showNotification('Error loading analytics data', 'error');
        })
        .finally(() => {
            hideLoadingState('analytics');
        });
}

function updateAnalyticsMetrics(metrics) {
    // Update metric cards
    document.getElementById('daily-active-users').textContent = metrics.daily_active_users || 0;
    document.getElementById('new-users-week').textContent = metrics.new_users_week || 0;
    document.getElementById('avg-session-duration').textContent = `${metrics.avg_session_duration || 0}`;
    document.getElementById('total-user-balance').textContent = `$${metrics.total_user_balance?.toFixed(2) || '0.00'}`;

    // Calculate and update KPIs
    const retentionRate = metrics.total_users > 0 ? ((metrics.daily_active_users / metrics.total_users) * 100).toFixed(1) : 0;
    const avgRevenuePerUser = metrics.total_users > 0 ? (metrics.total_user_balance / metrics.total_users).toFixed(2) : 0;

    document.getElementById('retention-rate').textContent = `${retentionRate}%`;
    document.getElementById('avg-revenue-per-user').textContent = `$${avgRevenuePerUser}`;
    document.getElementById('total-sessions').textContent = metrics.total_sessions?.toLocaleString() || '0';
    document.getElementById('completion-rate').textContent = `${metrics.completion_rate || 0}%`;
}

function initializeAnalyticsCharts(analyticsData) {
    // Ensure analytics section is visible before creating charts
    const analyticsSection = document.getElementById('analytics-section');
    if (!analyticsSection || analyticsSection.classList.contains('hidden')) {
        return;
    }

    // Destroy existing charts if they exist
    if (charts.activityChart) charts.activityChart.destroy();
    if (charts.revenueChart) charts.revenueChart.destroy();
    if (charts.playtimeChart) charts.playtimeChart.destroy();

    // Daily Active Users Chart
    const activityCtx = document.getElementById('activityChart');
    if (activityCtx) {
        charts.activityChart = new Chart(activityCtx.getContext('2d'), {
            type: 'line',
            data: {
                labels: analyticsData.user_activity.labels,
                datasets: [{
                    label: 'Active Users',
                    data: analyticsData.user_activity.data,
                    backgroundColor: 'rgba(59, 130, 246, 0.1)',
                    borderColor: 'rgb(59, 130, 246)',
                    borderWidth: 2,
                    fill: true,
                    tension: 0.4
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            }
        });
    }

    // Revenue Distribution Chart
    const revenueCtx = document.getElementById('revenueChart');
    if (revenueCtx && analyticsData.revenue_distribution.data.length > 0) {
        charts.revenueChart = new Chart(revenueCtx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: analyticsData.revenue_distribution.labels,
                datasets: [{
                    data: analyticsData.revenue_distribution.data,
                    backgroundColor: analyticsData.revenue_distribution.colors,
                    borderWidth: 2,
                    borderColor: '#fff'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                aspectRatio: 1,
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            font: {
                                size: 12
                            }
                        }
                    }
                },
                layout: {
                    padding: {
                        top: 10,
                        bottom: 10,
                        left: 10,
                        right: 10
                    }
                }
            }
        });
    }

    // Playtime Distribution Chart
    const playtimeCtx = document.getElementById('playtimeChart');
    if (playtimeCtx && analyticsData.playtime_distribution.data.length > 0) {
        charts.playtimeChart = new Chart(playtimeCtx.getContext('2d'), {
            type: 'bar',
            data: {
                labels: analyticsData.playtime_distribution.labels,
                datasets: [{
                    label: 'Users',
                    data: analyticsData.playtime_distribution.data,
                    backgroundColor: 'rgba(16, 185, 129, 0.6)',
                    borderColor: 'rgb(16, 185, 129)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            stepSize: 1
                        }
                    }
                }
            }
        });
    }
}

function showLoadingState(section) {
    // Add loading spinner or placeholder
    const sectionElement = document.getElementById(`${section}-section`);
    if (sectionElement) {
        sectionElement.style.opacity = '0.6';
    }
}

function hideLoadingState(section) {
    const sectionElement = document.getElementById(`${section}-section`);
    if (sectionElement) {
        sectionElement.style.opacity = '1';
    }
}

function showNotification(message, type = 'info') {
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `fixed top-4 right-4 p-4 rounded-md shadow-lg z-50 ${type === 'success' ? 'bg-green-500 text-white' :
        type === 'error' ? 'bg-red-500 text-white' :
            'bg-blue-500 text-white'
        }`;
    notification.textContent = message;

    document.body.appendChild(notification);

    // Remove after 3 seconds
    setTimeout(() => {
        document.body.removeChild(notification);
    }, 3000);
}
