<?php
// Note: Database class is injected via constructor, no direct require needed
// require_once '../config/database.php';

class User
{
    private $conn;
    private $table_name = "users";

    public $id;
    public $username;
    public $email;
    public $password_hash;
    public $device_id;
    public $total_playtime_minutes;
    public $current_balance;
    public $total_earned;
    public $total_withdrawn;
    public $admob_earnings;
    public $status;

    public function __construct($db)
    {
        $this->conn = $db;
    }

    // Create user
    public function create()
    {
        $query = "INSERT INTO " . $this->table_name . " 
                  SET username=:username, email=:email, password_hash=:password_hash, device_id=:device_id";

        $stmt = $this->conn->prepare($query);

        // Sanitize
        $this->username = htmlspecialchars(strip_tags($this->username));
        $this->email = htmlspecialchars(strip_tags($this->email));
        $this->device_id = htmlspecialchars(strip_tags($this->device_id));

        // Bind values
        $stmt->bindParam(":username", $this->username);
        $stmt->bindParam(":email", $this->email);
        $stmt->bindParam(":password_hash", $this->password_hash);
        $stmt->bindParam(":device_id", $this->device_id);

        if ($stmt->execute()) {
            $this->id = $this->conn->lastInsertId();
            return true;
        }
        return false;
    }

    // Read user by device ID
    public function readByDeviceId()
    {
        $query = "SELECT * FROM " . $this->table_name . " WHERE device_id = :device_id LIMIT 0,1";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':device_id', $this->device_id);
        $stmt->execute();

        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $this->id = $row['id'];
            $this->username = $row['username'];
            $this->email = $row['email'];
            $this->password_hash = $row['password_hash'];
            $this->total_playtime_minutes = $row['total_playtime_minutes'];
            $this->current_balance = $row['current_balance'];
            $this->total_earned = $row['total_earned'];
            $this->total_withdrawn = $row['total_withdrawn'];
            $this->admob_earnings = $row['admob_earnings'];
            $this->status = $row['status'];
            return true;
        }
        return false;
    }

    // Update balance
    public function updateBalance($amount)
    {
        $query = "UPDATE " . $this->table_name . " 
                  SET current_balance = current_balance + :amount,
                      total_earned = total_earned + :amount
                  WHERE id = :id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':amount', $amount);
        $stmt->bindParam(':id', $this->id);

        return $stmt->execute();
    }

    // Update playtime
    public function updatePlaytime($minutes)
    {
        $query = "UPDATE " . $this->table_name . " 
                  SET total_playtime_minutes = total_playtime_minutes + :minutes
                  WHERE id = :id";

        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':minutes', $minutes);
        $stmt->bindParam(':id', $this->id);

        return $stmt->execute();
    }

    // Get all users with pagination
    public static function getAllUsers($db, $offset = 0, $limit = 20)
    {
        $query = "SELECT id, username, email, device_id, total_playtime_minutes, 
                         current_balance, total_earned, total_withdrawn, admob_earnings, 
                         status, created_at 
                  FROM users 
                  ORDER BY created_at DESC 
                  LIMIT :limit OFFSET :offset";

        $stmt = $db->prepare($query);
        $stmt->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();

        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Get user statistics
    public static function getUserStats($db)
    {
        $query = "SELECT 
                    COUNT(*) as total_users,
                    SUM(current_balance) as total_balance,
                    SUM(total_earned) as total_earned,
                    SUM(total_withdrawn) as total_withdrawn,
                    AVG(total_playtime_minutes) as avg_playtime
                  FROM users";

        $stmt = $db->prepare($query);
        $stmt->execute();

        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}
