# Fierce Mobile Game API

A complete backend solution for Unity mobile games with user accounts, wallet system, withdrawal management, and AdMob integration.

## Features

### 🎮 Game Features
- User authentication via device ID
- Session-based playtime tracking
- Automatic earnings calculation
- Race completion bonuses
- Daily earning limits

### 💰 Wallet System
- Real-time balance tracking
- Multiple earning sources (gameplay + AdMob)
- Secure withdrawal requests
- Payment method support (Bank, PayPal, Mobile Money, Crypto)

### 📊 Admin Dashboard
- Modern, responsive interface
- Real-time statistics
- Withdrawal request management
- User management
- System settings configuration
- Analytics and reporting

### 🔗 Unity Integration
- Complete C# scripts provided
- Easy API integration
- Event-driven architecture
- Error handling and validation

## Quick Setup

1. **Extract to XAMPP/Apache Directory**
   ```
   c:\xampp8\htdocs\Project\fierce-mobile-api\
   ```

2. **Run Setup Script**
   ```
   http://localhost/fierce-mobile-api/setup.php
   ```

3. **Access Admin Panel**
   ```
   http://localhost/fierce-mobile-api/admin/
   Username: admin
   Password: admin123
   ```

## Project Structure

```
fierce-mobile-api/
├── config/
│   └── database.php          # Database configuration
├── models/
│   ├── User.php             # User model
│   ├── WithdrawalRequest.php # Withdrawal model
│   └── UserSession.php      # Session model
├── api/                     # Unity API endpoints
│   ├── auth.php            # User authentication
│   ├── start_session.php   # Start game session
│   ├── end_session.php     # End game session
│   ├── record_admob.php    # Record AdMob revenue
│   └── withdrawal.php      # Withdrawal requests
├── admin/                   # Admin dashboard
│   ├── index.html          # Main dashboard
│   ├── js/admin.js         # Dashboard JavaScript
│   └── api/                # Admin API endpoints
├── database/
│   └── schema.sql          # Database schema
├── unity_scripts/          # Unity integration scripts
│   ├── FierceGameAPI.cs    # Main API class
│   └── GameUI.cs           # UI management
└── setup.php               # One-click setup
```

## API Endpoints

### Authentication
- `POST /api/auth.php` - Authenticate user by device ID

### Session Management
- `POST /api/start_session.php` - Start gameplay session
- `POST /api/end_session.php` - End session and calculate earnings

### Revenue Tracking
- `POST /api/record_admob.php` - Record AdMob revenue

### Withdrawals
- `POST /api/withdrawal.php` - Request withdrawal
- `GET /api/withdrawal.php` - Get user's withdrawal history

## Unity Integration

### 1. Copy Scripts
Copy all files from `unity_scripts/` to your Unity project.

### 2. Setup API Component
```csharp
// Add FierceGameAPI component to a GameObject
var gameAPI = gameObject.AddComponent<FierceGameAPI>();

// Setup event listeners
gameAPI.OnUserAuthenticated.AddListener(OnUserAuthenticated);
gameAPI.OnSessionEnded.AddListener(OnSessionEnded);
gameAPI.OnError.AddListener(OnError);
```

### 3. Track Game Events
```csharp
// When race is completed
gameAPI.EndGameSession(true);

// When user watches rewarded ad
gameAPI.OnRewardedAdComplete();

// When user exits game
gameAPI.EndGameSession(false);
```

## Configuration

### Earning Settings (Configurable via Admin Panel)
- **Earning Rate**: $0.10 per minute of gameplay
- **Minimum Withdrawal**: $5.00
- **Daily Earning Limit**: $50.00
- **AdMob Revenue Share**: 70% to users
- **Race Completion Bonus**: 50% extra earnings

### Payment Methods Supported
- Bank Transfer
- PayPal
- Mobile Money
- Cryptocurrency

## Admin Features

### Dashboard
- Total users count
- Pending withdrawal requests
- Total payouts processed
- User balance overview
- Earnings and withdrawal charts

### Withdrawal Management
- View all withdrawal requests
- Filter by status (Pending, Approved, Completed, Rejected)
- Process requests with admin notes
- Automatic balance management

### User Management
- View all registered users
- User statistics (balance, earnings, playtime)
- User status management

### System Settings
- Configure earning rates
- Set withdrawal limits
- Adjust AdMob revenue sharing
- Update daily earning caps

## Security Features

- SQL injection protection via PDO prepared statements
- Input validation and sanitization
- XSS protection with HTML escaping
- Automatic balance verification
- Transaction logging

## Database Schema

### Key Tables
- `users` - User accounts and wallet data
- `user_sessions` - Gameplay session tracking
- `withdrawal_requests` - Withdrawal management
- `admob_earnings` - AdMob revenue tracking
- `system_settings` - Configurable parameters
- `activity_logs` - Action logging

## Production Deployment

### Security Checklist
1. Change default admin password
2. Update database credentials
3. Implement API authentication tokens
4. Enable HTTPS
5. Configure proper file permissions
6. Set up backup procedures

### Server Requirements
- PHP 7.4+
- MySQL 5.7+
- Apache/Nginx
- PDO MySQL extension

## Support & Customization

The system is designed to be easily customizable:

- Add new payment methods in `models/WithdrawalRequest.php`
- Modify earning calculations in `models/UserSession.php`
- Extend API endpoints in the `api/` directory
- Customize admin interface in `admin/`

## License

This project is provided as-is for game development purposes.

---

**Happy Gaming! 🎮**
